package commands

import (
	"context"
	"encoding/json"
	"log/slog"
	"os"
	"sync"

	"github.com/redis/go-redis/v9"
	"github.com/urfave/cli/v3"
)

var ImportCommand = cli.Command{
	Name:  "import",
	Usage: "import data ta Unity Redis",
	Flags: []cli.Flag{
		&cli.StringFlag{
			Name:     "host",
			Aliases:  []string{"h"},
			Usage:    "Redis URL",
			Required: true,
		},
		&cli.StringFlag{
			Name:     "source",
			Usage:    "source file to import data from",
			Required: true,
		},
		&cli.IntFlag{
			Name:     "database",
			Aliases:  []string{"db"},
			Value:    0,
			Usage:    "the database to connect",
			Required: false,
		},
		&cli.IntFlag{
			Name:     "concurrency",
			Value:    10,
			Usage:    "number of concurrent workers",
			Required: false,
		},
		&cli.BoolFlag{
			Name:     "force",
			Aliases:  []string{"f"},
			Usage:    "overwrite existing keys",
			Value:    false,
			Required: false,
		},
	},
	Action: func(ctx context.Context, cmd *cli.Command) error {
		redisURL := cmd.String("host")
		inputFile := cmd.String("source")
		database := cmd.Int("database")
		concurrency := cmd.Int("concurrency")
		overwrite := cmd.Bool("force")
		return importData(ctx, redisURL, inputFile, database, concurrency, overwrite)
	},
}

func importData(ctx context.Context, redisURL, inputFile string, database, concurrency int, overwrite bool) error {
	rdb := redis.NewClient(&redis.Options{
		Addr: redisURL,
		DB:   database,
	})
	defer rdb.Close()

	err := rdb.Ping(ctx).Err()
	if err != nil {
		return err
	}

	// Read the export file
	file, err := os.Open(inputFile)
	if err != nil {
		return err
	}
	defer file.Close()
	var fileExport FileExport

	decoder := json.NewDecoder(file)
	err = decoder.Decode(&fileExport)
	if err != nil {
		return err
	}
	imports := fileExport.Keys

	slog.Info("loaded keys from file", "count", len(imports))

	// Channels for distributing work
	importsChan := make(chan KeyExport, len(imports))
	var wg sync.WaitGroup
	var successCount, skipCount, errorCount int
	var mu sync.Mutex

	// Worker function to import key data
	worker := func() {
		defer wg.Done()
		for item := range importsChan {
			// Check if key exists (if not overwriting)
			if !overwrite {
				exists, err := rdb.Exists(ctx, item.Key).Result()
				if err != nil {
					slog.Warn("failed to check key existence", "key", item.Key, "error", err)
					mu.Lock()
					errorCount++
					mu.Unlock()
					continue
				}
				if exists > 0 {
					slog.Debug("skipping existing key", "key", item.Key)
					mu.Lock()
					skipCount++
					mu.Unlock()
					continue
				}
			}

			// Set value based on type
			var err error
			switch item.Type {
			case "string":
				strValue, ok := item.Value.(string)
				if !ok {
					slog.Warn("invalid string value", "key", item.Key)
					mu.Lock()
					errorCount++
					mu.Unlock()
					continue
				}
				err = rdb.Set(ctx, item.Key, strValue, 0).Err()

			case "list":
				listValue, ok := item.Value.([]interface{})
				if !ok {
					slog.Warn("invalid list value", "key", item.Key)
					mu.Lock()
					errorCount++
					mu.Unlock()
					continue
				}
				if len(listValue) > 0 {
					err = rdb.RPush(ctx, item.Key, listValue...).Err()
				}

			case "set":
				setValue, ok := item.Value.([]interface{})
				if !ok {
					slog.Warn("invalid set value", "key", item.Key)
					mu.Lock()
					errorCount++
					mu.Unlock()
					continue
				}
				if len(setValue) > 0 {
					err = rdb.SAdd(ctx, item.Key, setValue...).Err()
				}

			case "zset":
				zsetValue, ok := item.Value.([]interface{})
				if !ok {
					slog.Warn("invalid zset value", "key", item.Key)
					mu.Lock()
					errorCount++
					mu.Unlock()
					continue
				}
				// Convert to redis.Z format
				var members []redis.Z
				for _, v := range zsetValue {
					zMap, ok := v.(map[string]interface{})
					if !ok {
						continue
					}
					member, hasM := zMap["Member"]
					score, hasS := zMap["Score"]
					if hasM && hasS {
						scoreFloat, _ := score.(float64)
						members = append(members, redis.Z{
							Score:  scoreFloat,
							Member: member,
						})
					}
				}
				if len(members) > 0 {
					err = rdb.ZAdd(ctx, item.Key, members...).Err()
				}

			case "hash":
				hashValue, ok := item.Value.(map[string]interface{})
				if !ok {
					slog.Warn("invalid hash value", "key", item.Key)
					mu.Lock()
					errorCount++
					mu.Unlock()
					continue
				}
				if len(hashValue) > 0 {
					err = rdb.HSet(ctx, item.Key, hashValue).Err()
				}

			default:
				slog.Warn("unsupported key type", "key", item.Key, "type", item.Type)
				mu.Lock()
				errorCount++
				mu.Unlock()
				continue
			}

			if err != nil {
				slog.Warn("failed to set key", "key", item.Key, "error", err)
				mu.Lock()
				errorCount++
				mu.Unlock()
				continue
			}

			mu.Lock()
			successCount++
			mu.Unlock()
		}
	}

	// Start workers
	for range concurrency {
		wg.Add(1)
		go worker()
	}

	// Send imports to workers
	for _, item := range imports {
		importsChan <- item
	}
	close(importsChan)

	// Wait for all workers to complete
	wg.Wait()

	slog.Info("import completed",
		"imported", successCount,
		"skipped", skipCount,
		"errors", errorCount,
		"total", len(imports))
	return nil
}
