package commands

import (
	"context"
	"fmt"
	"os"
	"path/filepath"

	"github.com/urfave/cli/v3"
)

var DiffCommand = cli.Command{
	Name:  "diff",
	Usage: "compare two Redis instances or export files",
	UsageText: "diff <source1> <source2>\n\n" +
		"   Sources can be:\n" +
		"     redis://host:port/db  - Redis instance\n" +
		"     file://path          - JSON export file\n\n" +
		"   Examples:\n" +
		"     diff redis://localhost:6379/0 redis://prod:6379/1\n" +
		"     diff file://export.json redis://localhost:6379/0\n" +
		"     diff file://backup1.json file://backup2.json",
	Arguments: []cli.Argument{
		&cli.StringArgs{
			Name:      "source",
			Min:       2,
			Max:       -1,
			UsageText: "source1 source2",
			Config: cli.StringConfig{
				TrimSpace: true,
			},
		},
	},
	Action: func(ctx context.Context, cmd *cli.Command) error {
		sources := cmd.StringArgs("source")
		if len(sources) != 2 {
			return fmt.Errorf("exactly two sources must be provided")
		}

		rawSourceA := sources[0]
		rawSourceB := sources[1]

		sourceA, err := ParseSource(rawSourceA)
		if err != nil {
			return fmt.Errorf("source 1: %w", err)
		}

		sourceB, err := ParseSource(rawSourceB)
		if err != nil {
			return fmt.Errorf("source 2: %w", err)
		}

		tmpDir, err := os.MkdirTemp("/tmp", "redis-diff-")
		if err != nil {
			return fmt.Errorf("creating temp dir: %w", err)
		}
		defer os.RemoveAll(tmpDir)

		if sourceA.Type == SourceTypeRedis {
			err := exportData(ctx, sourceA.Host, "*", filepath.Join(tmpDir, "sourceA.json"), sourceA.Database, 10)
			if err != nil {
				return fmt.Errorf("exporting source 1: %w", err)
			}
			sourceA.Type = SourceTypeFile
			sourceA.FilePath = filepath.Join(tmpDir, "sourceA.json")
		}
		if sourceB.Type == SourceTypeRedis {
			err := exportData(ctx, sourceB.Host, "*", filepath.Join(tmpDir, "sourceB.json"), sourceB.Database, 10)
			if err != nil {
				return fmt.Errorf("exporting source 2: %w", err)
			}
			sourceB.Type = SourceTypeFile
			sourceB.FilePath = filepath.Join(tmpDir, "sourceB.json")
		}
		return diffFiles(ctx, sourceA.FilePath, sourceB.FilePath)
	},
}

func diffFiles(ctx context.Context, fileA, fileB string) error {
	return nil
}
