package commands

import (
	"fmt"
	"os"
	"strconv"
	"strings"
)

type KeyExport struct {
	Key   string `json:"key"`
	Type  string `json:"type"`
	Value any    `json:"value"`
}

type FileExport struct {
	RedisInstance string      `json:"redis_instance"`
	Database      int         `json:"database"`
	DateCreated   string      `json:"date_created"`
	Keys          []KeyExport `json:"keys"`
}

type SourceType int

const (
	SourceTypeRedis SourceType = iota
	SourceTypeFile
)

type Source struct {
	Type     SourceType
	Host     string // For Redis: host:port
	Database int    // For Redis: database number
	FilePath string // For File: path to file
}

// ParseSource parses a source string in format:
// - redis://host:port/db (e.g., redis://localhost:6379/0)
// - file://path/to/file.json (e.g., file://export.json)
func ParseSource(input string) (*Source, error) {
	// Check for redis:// prefix
	if strings.HasPrefix(input, "redis://") {
		return parseRedisSource(input)
	}

	// Check for file:// prefix
	if strings.HasPrefix(input, "file://") {
		return parseFileSource(input)
	}

	return nil, fmt.Errorf("invalid source format: must start with 'redis://' or 'file://'")
}

func parseRedisSource(input string) (*Source, error) {
	// Remove redis:// prefix
	rest := strings.TrimPrefix(input, "redis://")

	// Split by / to separate host:port from database
	parts := strings.Split(rest, "/")
	if len(parts) != 2 {
		return nil, fmt.Errorf("invalid redis source format: expected 'redis://host:port/db'")
	}

	hostPort := parts[0]
	dbStr := parts[1]

	// Validate host:port has a colon
	if !strings.Contains(hostPort, ":") {
		return nil, fmt.Errorf("invalid redis host: expected 'host:port' format")
	}

	// Parse database number
	db, err := strconv.Atoi(dbStr)
	if err != nil {
		return nil, fmt.Errorf("invalid database number: %s", dbStr)
	}

	if db < 0 {
		return nil, fmt.Errorf("database number must be non-negative")
	}

	return &Source{
		Type:     SourceTypeRedis,
		Host:     hostPort,
		Database: db,
	}, nil
}

func parseFileSource(input string) (*Source, error) {
	// Remove file:// prefix
	filePath := strings.TrimPrefix(input, "file://")

	if filePath == "" {
		return nil, fmt.Errorf("invalid file source: path cannot be empty")
	}

	// Check if file exists
	if _, err := os.Stat(filePath); os.IsNotExist(err) {
		return nil, fmt.Errorf("file does not exist: %s", filePath)
	}

	return &Source{
		Type:     SourceTypeFile,
		FilePath: filePath,
	}, nil
}

// String returns a human-readable representation of the source
func (s *Source) String() string {
	switch s.Type {
	case SourceTypeRedis:
		return fmt.Sprintf("redis://%s/%d", s.Host, s.Database)
	case SourceTypeFile:
		return fmt.Sprintf("file://%s", s.FilePath)
	default:
		return "unknown source"
	}
}
