package commands

import (
	"context"
	"encoding/json"
	"log/slog"
	"os"
	"sync"
	"time"

	"github.com/redis/go-redis/v9"
	"github.com/urfave/cli/v3"
)

var ExportCommand = cli.Command{
	Name:  "export",
	Usage: "export data from Unity Redis",
	Flags: []cli.Flag{
		&cli.StringFlag{
			Name:     "host",
			Aliases:  []string{"h"},
			Usage:    "Redis URL",
			Required: true,
		},
		&cli.StringFlag{
			Name:     "pattern",
			Value:    "*",
			Usage:    "key pattern to match",
			Required: false,
		},
		&cli.StringFlag{
			Name:     "output",
			Value:    "export.json",
			Usage:    "output file for exported data",
			Required: false,
		},
		&cli.IntFlag{
			Name:     "database",
			Aliases:  []string{"db"},
			Value:    0,
			Usage:    "the database to connect",
			Required: false,
		},
		&cli.IntFlag{
			Name:     "concurrency",
			Value:    10,
			Usage:    "number of concurrent workers",
			Required: false,
		},
	},
	Action: func(ctx context.Context, cmd *cli.Command) error {
		redisURL := cmd.String("host")
		keyPattern := cmd.String("pattern")
		outputFile := cmd.String("output")
		database := cmd.Int("database")
		concurrency := cmd.Int("concurrency")
		return exportData(ctx, redisURL, keyPattern, outputFile, database, concurrency)
	},
}

func exportData(ctx context.Context, redisURL, keyPattern, outputFile string, database, concurrency int) error {
	rdb := redis.NewClient(&redis.Options{
		Addr: redisURL,
		DB:   database,
	})
	defer rdb.Close()

	err := rdb.Ping(ctx).Err()
	if err != nil {
		return err
	}

	// Use SCAN to iterate over keys matching the pattern
	var cursor uint64
	var keys []string
	for {
		var batch []string
		var err error
		batch, cursor, err = rdb.Scan(ctx, cursor, keyPattern, 100).Result()
		if err != nil {
			return err
		}
		keys = append(keys, batch...)
		if cursor == 0 {
			break
		}
	}
	slog.Info("found keys", "count", len(keys))

	// Export data structure to hold key dumps

	// Channels for distributing work and collecting results
	keysChan := make(chan string, len(keys))
	resultsChan := make(chan KeyExport, len(keys))
	var wg sync.WaitGroup

	// Worker function to fetch key data
	worker := func() {
		defer wg.Done()
		for key := range keysChan {
			// Get key type
			keyType, err := rdb.Type(ctx, key).Result()
			if err != nil {
				slog.Warn("failed to get key type", "key", key, "error", err)
				continue
			}

			var value any

			// Get value based on type
			switch keyType {
			case "string":
				value, err = rdb.Get(ctx, key).Result()
			case "list":
				value, err = rdb.LRange(ctx, key, 0, -1).Result()
			case "set":
				value, err = rdb.SMembers(ctx, key).Result()
			case "zset":
				value, err = rdb.ZRangeWithScores(ctx, key, 0, -1).Result()
			case "hash":
				value, err = rdb.HGetAll(ctx, key).Result()
			default:
				slog.Warn("unsupported key type", "key", key, "type", keyType)
				continue
			}

			if err != nil {
				slog.Warn("failed to get value for key", "key", key, "error", err)
				continue
			}

			resultsChan <- KeyExport{
				Key:   key,
				Type:  keyType,
				Value: value,
			}
		}
	}

	// Start workers
	for range concurrency {
		wg.Add(1)
		go worker()
	}

	// Send keys to workers
	for _, key := range keys {
		keysChan <- key
	}
	close(keysChan)

	// Wait for all workers to complete and close results channel
	go func() {
		wg.Wait()
		close(resultsChan)
	}()

	// Collect results
	var exports []KeyExport
	for export := range resultsChan {
		exports = append(exports, export)
	}
	fileExport := FileExport{
		RedisInstance: redisURL,
		Database:      database,
		DateCreated:   time.Now().Format(time.RFC3339),
		Keys:          exports,
	}

	// Write to output file
	file, err := os.Create(outputFile)
	if err != nil {
		return err
	}
	defer file.Close()

	encoder := json.NewEncoder(file)
	encoder.SetIndent("", "  ")
	err = encoder.Encode(fileExport)
	if err != nil {
		return err
	}

	slog.Info("export completed", "keys_exported", len(exports), "output", outputFile)
	return nil
}
